(* ::Subsubsection:: *)
(*SchmidtCamera package*)

(* ::Text:: *)
(*This impements the Schmidt Camera optical system*)

(* ::Text:: *)
(*Chapter*)

(* ::Text:: *)
(*This file should be in the following path*)

(* ::Text:: *)
(*FileNameJoin[{$UserBaseDirectory, "Applications/GeometricOptics/SchmidtCamera"}]*)

BeginPackage["GeometricOptics`Packages`SchmidtCamera`", {"GeometricOptics`"}]

Options[SchmidtCamera] = {OutputType->"Report", OutputLevel->"Full"};
(* 	OutputType can be 
					"Report", generates a new report (notebook) for each call to the function
					"Print", prints results in the current position of the evaluating notebook
					"Basic", gives the list of results
					"Values" or any other directive, gives the list of values (no names of variables)
	OutputLevel can be
					"Minimal", gives only output values, those calculated by the function
					"Full", gives the list of input parameters and output values
*)
SchmidtCamera::usage="SchmidtCamera calculates parameters for a Maksutov Camera optical system.";

SchmidtCameraInputPanel::usage = "SchmidtCameraInputPanel gives the GUI panel to work with SchmidtCamera";

$SchmidtCameraInputVariables = {"f1", "ind", "thick", "diam", "\[Alpha]", "\[Theta]"};

Begin["`Private`"]

SchmidtCamera[{f1_, ind_, thick_, diam_, alfa_, theta_}, opts___]:= SchmidtCamera[f1, ind, thick, diam, alfa, theta, opts];

SchmidtCamera[f1_, ind_, thick_, diam_, alfa_, theta_, OptionsPattern[]] := 
 Module[{a4, r, Rc, sol, fun, sag, outputs, inPanel, y, outPanel},
 
		If[ArgumentsQ["SchmidtCamera", {$SchmidtCameraInputVariables, {f1, ind, thick, diam, alfa, theta}}],
			AppendTo[$ExamplesStack, 
					 <|"PackageID" -> 6, "PackageName" -> "SchmidtCamera", "Arguments" ->  <|"f1" -> f1, "ind" -> ind, "thick" -> thick, "diam" -> diam, "\[Alpha]" -> alfa, "\[Theta]" -> theta|>|>];
			$ExamplesStack = DeleteExampleDuplicates[$ExamplesStack];
			(* Value of aspheric coefficient *)
			a4 = 1/(32 (-1 + ind[[1, 2]]) f1^3);
			r = diam/2;
			Rc = -8 f1^3 (ind[[1, 2]] - 1)/(alfa ^2 r^2);
			(* Aberrations of the combination *)
			sol = TotalAberrations[{10^10, Rc, -2 f1}, {thick, 2 f1}, ind, {0, {a4, as}, 0}, diam/2, 0, 0, -Infinity, y1, theta, {\[Lambda]1, \[Lambda]2, \[Lambda]3}];
			
			fun = 1/(2 Rc)*Global`y^2 + a4*Global`y^4;
			sag = fun /. {Global`y -> alfa diam/2};
						
			outputs = {{"Corrector equation", fun},
						{"Corrector plot", Plot[fun, {Global`y, -diam/2, diam/2}, AspectRatio -> 1/3, AxesOrigin -> {0, 0}]},
						{"Corrector sagitta", sag},
						{"Curvature radius of the second corrector surface", Rc},
						{"Curvature radius of the main mirror", -2 f1},
						{"Image height", GOimageHeight[[1, GOn]]},
						{"Focal lenght", GOfocalLength[[1]]},
						{"Spherical aberration", GOaberration[[1]]},
						{"Sagittal coma", GOcoma[[1]]},
						{"Astigmatism", GOastigmatism[[1]]},
						{"Axial chromatic aberration", GOdistancegauss[[2, GOn]] - GOdistancegauss[[3, GOn]]}};

			(* defines the two panels, input parameters and output values *)
			inPanel = Grid[{{"Focal length of the primary mirror" , "f1", f1},
							{"List of the refractive indices", "ind", Grid@ind},
							{"Thickness of the corrector", "thick", thick},
							{"Diameter of primary mirror", "diam", diam},
							{"Form factor of the second surface of the corrector", alfa},
							{"Field angle  in degree", "theta", theta}},
							Alignment -> {{Left, Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"InputParameterBottom"}];
  
			outPanel = Grid[outputs, 
							Alignment -> {{Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"OutputValueBottom"}];
				
			(* generates the type of output required *)
			Switch[OptionValue[OutputType],
					"Report",
					GenerateDocument[TemplateApply[$ReportTemplate, 
										Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "SchmidtCamera" &], "Description"], 
												"date" -> DateString[], 
												"function" -> "SchmidtCamera", 
												"outputlevel" -> OptionValue[OutputLevel],
												"inPanel" -> inPanel, 
												"outPanel" -> outPanel |>]]];,

					"Print",
					CellPrint[TextCell[TemplateApply[$PrintTemplate, 
											Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "SchmidtCamera" &], "Description"], 
													"date" -> DateString[], 
													"function" -> "SchmidtCamera", 
													"outputlevel" -> OptionValue[OutputLevel],
													"inPanel" -> inPanel, 
													"outPanel" -> outPanel |>]], "Text"]];,
					"Basic",
					CellPrint[TextCell[
					TemplateApply[$BasicTemplate, 
									Join[<| "outputlevel" -> OptionValue[OutputLevel],
											"inputs" -> {{"f1" , f1},
														 {"ind", ind},
														 {"thick", thick},
														 {"diam", diam},
														 {"\[Alpha]" , alfa},
														 {"\[Theta]", theta}},
											"outputs" -> outputs |>]], "Output"]],
					_,
					CellPrint[TextCell[DeleteCases[outputs[[All, 2]], Alternatives["", Style[___]]], "Output"]]],

							
			(* Arguments are not correct *)
			MessageDialog["SchmidtCamera not executed, the number or the type of the arguments may be incorrect.", WindowTitle->"Warning: example not generated"];]];

SchmidtCameraInputPanel[]:=
DynamicModule[{outputtype = "Report", package = "SchmidtCamera"},
	examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
						Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
    examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, examplesAll];
	example = "Browse...";
	Panel[Column[{	DynamicWrapper[Style[NameFromPackageName[package] <> " Input Panel", "Subsection"], 
									If[NumericQ[example], {f1, ind, thick, diam, alfa, theta} = ReplaceAll[$SchmidtCameraInputVariables, examplesAll[[example]]]]],
					Style["Insert values for each argument, then use Evaluate to run "<>package<>" function", "Text"],
					Grid[{{"Focal length of the primary mirror" , "f1", Tooltip[InputField[Dynamic[f1], Alignment -> Center], "Insert the value of focal length of the primary mirror"]},
						  {"List of the refractive indices", "ind", Tooltip[InputField[Dynamic[ind], Alignment -> Center], "Insert the list of the refractive indices"]},
						  {"Thickness of the corrector", "thick", Tooltip[InputField[Dynamic[thick], Alignment -> Center], "Insert the thickness of the corrector"]},
						  {"Diameter of the primary mirror", "diam", Tooltip[InputField[Dynamic[diam], Alignment -> Center], "Input the value of diam"]},
						  {"Form factor of the second surface of the corrector", "alfa", Tooltip[InputField[Dynamic[alfa], Alignment -> Center], "Insert the \[Alpha] coefficient"]},
						  {"Field angle in degrees", "\[Theta]", Tooltip[InputField[Dynamic[theta], Alignment -> Center], "Insert the value of \[Theta]"]}},
						Spacings -> {1, 0},
						Alignment -> {{Left, Left, Right}, Center}, 
						Dividers -> Center, 
						FrameStyle -> LightGray],
					OpenerView[{"Load an example from the archives (current session and saved DB)",
								Row[{Dynamic@PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], If[examples === {}, "No example saved", "Browse..."], FrameMargins -> 3, Alignment -> Center],
									 Spacer[5],
									 Button["Update list", (examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
																				Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
															examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], 
																						Alignment -> Center, 
																						Spacings -> {1, 1}, 
																						Dividers -> Center], 
																						FrameStyle -> LightGray]&, examplesAll];
															example = "Browse..."), Method -> "Queued"]}]}, 
								Alignment -> Center, 
								Spacings -> 1.5],
					Row[{"Define the type of output to generate", 
						 Spacer[5],
						 RadioButtonBar[Dynamic[outputtype], {"Report" -> Tooltip["Report", "Generates a new notebook reporting a summary of the calculation"], 
															  "Print" -> Tooltip["Print", "Print the table of the calculation done inside the current notebook"], 
															  "Basic" -> Tooltip["Basic", "Generate a list of computed output with label"], 
															  "Values" -> Tooltip["Values", "Return only the list of output values"]}]}],
					Row[{Button["Evaluate", ToExpression[package][Apply[Sequence, {f1, ind, thick, diam, alfa, theta}], OutputType -> outputtype], Method -> "Queued"],
						 Button["Clear all", Map[Clear, Unevaluated[{f1, ind, thick, diam, alfa, theta}]]]}]}, 
				Spacings -> 2, 
				Alignment -> Center],
	BaseStyle -> {InputFieldBoxOptions -> {FieldSize -> {15, 1}}}]];
			
  
  End[]
  EndPackage[]